function raster_plot = raster(data_array,varargin)
%This function will produce an image that is a true raster plot of the
%entered data array, with each row being a data set.
%Syntax: raster_plot = raster(data_array,'psize',[],'border','',...
%               'border_size',[]);
%Input: data_array = A 2D array with each row being a data set
%       'psize' = pixel size, the number of pixels x,y that makes up a single
%       point in the raster array.  Default = [5 10]
%       'border' = The edge color for each point in the raster array.
%       Default = 'black'.  Other choices: 'white','none'=border_size 0
%       'border_size' = The pixel width of the border.  Default = 2
%Output: raster_plot = the true color image that is the raster plot.  A x by y by
%               3 matrix.

%parse inputs
[psize,border,bsize] = parse(varargin);
%set defaults
[a,b] = size(data_array);
data_array = double(data_array);    %convert of double for accuracy
minimum = min(min(data_array));     %get the min and max of the array
maximum = max(max(data_array));
plot_y = a*psize(1,2);      %calculate the plot y axis size
plot_x = b*psize(1,1);      %calculate the plot x axis size

%normalize data_array to between 0 and 1, if it is not already, so it is easier to map
if minimum<0 || maximum>1
    data_array = data_array-minimum;
    maximum = maximum-minimum;
    data_array = data_array./maximum;
end

%initialize the plot & work in borders if necessary
if bsize ~= 0
    raster_plot = zeros((plot_y+(bsize*a+bsize)),((plot_x+(bsize*b+bsize))));  %Initialize plot with borders
    xborder = meshgrid(linspace(border,border,(plot_x+bsize*b+bsize)),1:bsize);   %create the x border
    raster_plot(1:bsize,:) = xborder;      %Insert the top border
else
    raster_plot = zeros(plot_y,plot_x);    %Initialize plot without borders
end

%create the raster plot
row_index = bsize+1;      %initialize the row index for the plot
for i = 1:a         %the rows
    %row_index = row_index+1;    %Step the row index to the right position
    col_index = 1;      %The column index for the plot
    %put in the first column border
    raster_plot(row_index,col_index:bsize) = linspace(border,border,bsize);
    col_index = col_index+bsize;        %stp to the correct location for col_index
    for j = 1:b     %the columns
        %create the x dimension of the pixel size
        if bsize ~= 0   %border
            raster_plot(row_index,col_index:(col_index+(psize(1,1)+bsize)-1)) = [...
                linspace(data_array(i,j),data_array(i,j),psize(1,1)) linspace(border,border,bsize)];
            col_index = col_index+(psize(1,1)+bsize);                
        else
            raster_plot(row_index,col_index:(col_index+psize(1,1)-1)) = linspace(data_array(i,j),data_array(i,j),psize(1,1));
            col_index = col_index+psize(1,1);
        end
    end
    l = 1;  %reset the y pointer
    for l = 1:(psize(1,2)-1)    %create the y dimension of the data point
        raster_plot(row_index+l,:) = raster_plot(row_index,:);    %copy the rows psize(1,2)-1
    end
    row_index = row_index+psize(1,2); %increment the row_index
    %border
    if bsize ~=0
        raster_plot(row_index:row_index+(bsize-1),:) = xborder; %add the bottom border
        row_index = row_index+bsize;    %move the row_index to the proper location
    end
end


%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [psize,border,bsize] = parse(input)
%set defaults
psize = [5 10];
border = 0;   %black
bsize = 2;

%Parse the input
if ~isempty(input)
    a = size(input,2);
    for i = 1:2:a
        value = input{1,i+1};
        switch input{1,i}
            case 'psize'
                if isnumeric(value)
                    psize = value;
                else
                    error(['Error: You did not enter the proper value for the psize property']);
                end
            case 'border'
                if isnumeric(value)
                    if max(value)>1
                        error(['Error: the value for border color must be between 0 and 1']);
                    else
                        border = value;     %take the numeric value of the color
                    end
                else
                    switch value
                        case 'black'
                            border = 0;     %black border
                        case 'white'
                            border = 1;     %white border: function to find out this value automatically?
                        case 'none'
                            bsize = 0;
                    end
                end
            case 'border_size'
                if isnumeric(value) & bsize ~=0     %the 'none' parameter for border supercedes this
                    bsize = value;
                end
        end
    end
end